<?php

namespace App\Services;

use App\Http\Requests\CostRequest;
use App\Http\Requests\CostUpdateRequest;
use App\Models\{Cost, CostItem};
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class CostService
{
    public function __construct(
        private UploadService $upload_service,
    ) {
    }

    public function find($id)
    {
        return Cost::find($id);
    }

    public function findDetail($id)
    {
        return CostItem::find($id);
    }

    public function list()
    {
        $user = auth()->user();

        $data = Cost::with('items')
        ->withParameters([
            'id_user' => $user->id
        ])
        ->orderby('id', 'desc');
        $data = $data->latest()->paginate(10);

        $data->getCollection()->transform(function ($item) {

            $show['id']                  = $item->id;
            $show['purchase_date']       = translatedFormat($item->purchase_date, 'd/m/Y');
            $show['total_cost']          = $item->items?->sum('subtotal');

            return $show;
        });

        return $data;
    }

    public function history_month()
    {
        $user = auth()->user();

        $_current  = Carbon::now()->format('Y-m');
        $_previous = Carbon::now()->subMonth()->format('Y-m');

        $data = CostItem::with('cost')
        ->withParameters([
            'id_user' => $user->id,

        ])
        ->orderby('id', 'desc')
        ->groupby('id_category')
        ->get()->map(function ($item) use($_current, $_previous, $user){

            $_t_current  = CostItem::withParameters(['id_user' => $user->id, 'id_category' => $item->id_category, 'start_date' => $_current])->sum('subtotal');
            $_t_previous = CostItem::withParameters(['id_user' => $user->id, 'id_category' => $item->id_category, 'start_date' => $_previous])->sum('subtotal');

            $show['title'] = $item->category?->title;
            $show['color'] = 'default';
            $show['data']  = [$_t_previous, $_t_current];

            return $show;
        });

        $_all = [
            'title' => 'Grand Total',
            'color' => 'red',
            'data' => [
                CostItem::withParameters(['id_user' => $user->id, 'start_date' => $_previous])->sum('subtotal'),
                CostItem::withParameters(['id_user' => $user->id, 'start_date' => $_current])->sum('subtotal'),
            ]
        ];

        return collect($data)->push($_all);
    }

    public function create(CostRequest $request)
    {
        DB::beginTransaction();

        $user = auth()->user();

        $id = $this->get_id($user);

        $upload_nota = $this->upload_service->uploadImage($request->upload_nota, 'upload_nota');
        $upload_proof_of_payment = $this->upload_service->uploadImage($request->upload_proof_of_payment, 'upload_proof_of_payment');

        $_cost = Cost::create(array_merge($request->all(), [
            'id'            => $id,
            'id_user'       => $user->id,
            'upload_nota'   => $upload_nota,
            'upload_proof_of_payment' => $upload_proof_of_payment,
        ]));

        $costs = $request->costs;

        foreach ($costs as $cost) {

            CostItem::create([
                'id_op_pengeluaran' => $_cost->id,
                'id_category'       => $cost['id_category'],
                'title'             => $cost['title'],
                'description'       => $cost['description'],
                'qty'               => $cost['qty'],
                'unit'              => $cost['unit'],
                'subtotal'          => $cost['subtotal'],
            ]);
        }

        DB::commit();

        return true;
    }

    public function update(CostUpdateRequest $request)
    {
        DB::beginTransaction();
        
        $find = $this->find($request->id);

        if ($find) {
            $upload_nota = $request->hasFile('upload_nota') ? $this->upload_service->uploadImage($request->upload_nota, 'upload_nota'): $find->upload_nota;
            $upload_proof_of_payment = $request->hasFile('upload_proof_of_payment') ? $this->upload_service->uploadImage($request->upload_proof_of_payment, 'upload_proof_of_payment'): $find->upload_proof_of_payment;

            $find->update(array_merge($request->all(), [
                'upload_nota' => $upload_nota,
                'upload_proof_of_payment'    => $upload_proof_of_payment,
            ]));

            $costs = $request->costs;

            $_id = [];

            foreach ($costs as $cost) {
                if ($cost['id'] != "") {
                    $_item = $this->findDetail($cost['id']);

                    $_id[] = $cost['id'];
    
                    $_item->update([
                        'id_category'       => $cost['id_category'],
                        'title'             => $cost['title'],
                        'description'       => $cost['description'],
                        'qty'               => $cost['qty'],
                        'unit'              => $cost['unit'],
                        'subtotal'          => $cost['subtotal'],
                    ]);

                }
            }

            CostItem::whereNotIn('id', $_id)->where('id_op_pengeluaran', $find->id)->delete();

            foreach ($costs as $cost) {
                if ($cost['id'] == "") {
        
                    CostItem::create([
                        'id_op_pengeluaran' => $find->id,
                        'id_category'       => $cost['id_category'],
                        'title'             => $cost['title'],
                        'description'       => $cost['description'],
                        'qty'               => $cost['qty'],
                        'unit'              => $cost['unit'],
                        'subtotal'          => $cost['subtotal'],
                    ]);

                }
            }
        
        }
        DB::commit();

        return true;
        
    }

    public function delete($id)
    {
        $find = $this->find($id);

        if ($find) {
            $find->delete();
        }

        return true;
    }

    private function get_id($user)
    {
        $_cost = Cost::where('id_user', $user->id)->latest()->withTrashed()->first();

        $_id = (int) substr($_cost?->id, 18) ?? 0;

        return $user->code . 'OPX' . Carbon::now()->format('my') . ($_id + 1);
    }

    
}
