<?php

namespace App\Services;

use App\Http\Requests\ShoppingRequest;
use App\Http\Requests\ShoppingUpdateRequest;
use App\Models\{POItem, Shopping, ShoppingItem};
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class ShoppingService
{
    public function __construct(
        private UploadService $upload_service,
        private ProductService $product_service,
    ) {
    }

    public function find($id)
    {
        return Shopping::find($id);
    }

    public function findDetail($id)
    {
        return ShoppingItem::find($id);
    }

    public function list($request)
    {
        $user = auth()->user();

        $data = Shopping::with('vendor')
        ->withParameters(array_merge($request->toArray(), [
            'id_user' => $user->id
        ]))
        ->orderby('id', 'desc');
        $data = $data->latest()->paginate(10);

        $data->getCollection()->transform(function ($item) {

            $show['id']                  = $item->id;
            $show['purchase_date']       = translatedFormat($item->purchase_date, 'd/m/Y');
            $show['vendor']              = $item->vendor?->name_vendor;
            $show['total_product']       = count($item->items);

            return $show;
        });

        return $data;
    }

    public function create(ShoppingRequest $request)
    {
        DB::beginTransaction();

        $user = auth()->user();

        $id = $this->get_id($user);

        $transfer_photo = $request->hasFile('transfer_photo') ? $this->upload_service->uploadImage($request->transfer_photo, 'proof_of_payment') : null;
        $payment_photo = $request->hasFile('payment_photo') ? $this->upload_service->uploadImage($request->payment_photo, 'shopping_note') : null;

        $shopping = Shopping::create(array_merge($request->all(), [
            'id'      => $id,
            'id_user' => $user->id,
            'proof_of_payment' => $transfer_photo,
            'shopping_note'    => $payment_photo,
        ]));

        $products = $request->products;

        foreach ($products as $product) {
            $find = $this->product_service->find($product['id_product']);

            $_arrayproduct = [
                'title' => $find?->title,
                'sku' => $find?->sku,
                'category' => $find?->category?->title,
                'unit' => $find?->unit?->title,
                'package' => $find?->package?->title,
            ];

            ShoppingItem::create([
                'id_shopping'   => $shopping->id,
                'id_product'    => $product['id_product'],
                'product'       => $_arrayproduct,
                'price'         => $product['price'],
                'qty'           => $product['qty'],
                'subtotal'      => $product['subtotal'],
                'description'   => $product['description'],
            ]);
        }

        DB::commit();

        return true;
    }

    public function update(ShoppingUpdateRequest $request)
    {
        DB::beginTransaction();
        
        $find = $this->find($request->id);

        if ($find) {
            $transfer_photo = $request->hasFile('transfer_photo') ? $this->upload_service->uploadImage($request->transfer_photo, 'proof_of_payment'): $find->proof_of_payment;
            $payment_photo = $request->hasFile('payment_photo') ? $this->upload_service->uploadImage($request->payment_photo, 'shopping_note'): $find->shopping_note;

            $find->update(array_merge($request->all(), [
                'proof_of_payment' => $transfer_photo,
                'shopping_note'    => $payment_photo,
            ]));

            $products = $request->products;

            $_id = [];

            foreach ($products as $product) {
                if ($product['id'] != "") {
                    $_item = $this->findDetail($product['id']);

                    $_id[] = $product['id'];

                    $_product = $this->product_service->find($product['id_product']);

    
                    $_arrayproduct = [
                        'title' => $_product?->title,
                        'sku' => $_product?->sku,
                        'category' => $_product?->category?->title,
                        'unit' => $_product?->unit?->title,
                        'package' => $_product?->package?->title,
                    ];
    
                    $_item->update([
                        'product'       => $_arrayproduct,
                        'qty'           => $product['qty'],
                        'subtotal'      => $product['subtotal'],
                        'description'   => $product['description'],
                    ]);

                }
            }

            ShoppingItem::whereNotIn('id', $_id)->where('id_shopping', $find->id)->delete();

            foreach ($products as $product) {
                if ($product['id'] == "") {
                    $_product = $this->product_service->find($product['id_product']);

                    $_arrayproduct = [
                        'title' => $_product?->title,
                        'sku' => $_product?->sku,
                        'category' => $_product?->category?->title,
                        'unit' => $_product?->unit?->title,
                        'package' => $_product?->package?->title,
                    ];
        
                    ShoppingItem::create([
                        'id_shopping'   => $find->id,
                        'id_product'    => $product['id_product'],
                        'product'       => $_arrayproduct,
                        'qty'           => $product['qty'],
                        'subtotal'      => $product['subtotal'],
                        'description'   => $product['description'],
                    ]);

                }
            }
        
        }
        DB::commit();

        return true;
        
    }

    public function stock($request)
    {
        $user = auth()->user();

        $date = $request->purchase_date ?? Carbon::now()->toDateString();

        $data = ShoppingItem::with('shopping')
                    ->withParameters(array_merge($request->toArray(), ['id_user' => $user->id]))
                    ->groupby('id_product');

        $data = $data->latest()->paginate(20);

        $data->getCollection()->transform(function ($item) use($date) {

            $_qty = $this->sumQtybyIdProduct($item->id_product, $item->shopping?->id_user, $date);

            $_productValue = $this->getProductValue($item->id_product, $item->shopping?->id_user, $date);

            $show['product']             = $item->product;
            $show['qty']                 = (int) $_qty;
            $show['product_value']       = $_productValue;
            $show['min_price']           = (int) ShoppingItem::withParameters(['id_user' => $item->shopping?->id_user, 'purchase_date' => $date, 'id_product' => $item->id_product])->min('price');
            $show['max_price']           = (int) ShoppingItem::withParameters(['id_user' => $item->shopping?->id_user, 'purchase_date' => $date, 'id_product' => $item->id_product])->max('price');
            $show['total_stock']         = $_productValue * $_qty;

            return $show;
        });

        return $data;
    }

    public function delete($id)
    {
        $find = $this->find($id);

        if ($find) {
            $find->delete();
        }

        return true;
    }

    private function get_id($user)
    {
        $_shop = Shopping::where('id_user', $user->id)->latest()->withTrashed()->first();

        $_id = (int) substr($_shop?->id, 18) ?? 0;

        return $user->code . 'PUR' . Carbon::now()->format('my') . ($_id + 1);
    }

    private function sumQtybyIdProduct($product_id, $user_id, $date)
    {
        return ShoppingItem::withParameters(['id_user' => $user_id, 'purchase_date' => $date, 'id_product' => $product_id])->sum('qty');
    }

    private function getProductValue($product_id, $user_id, $date)
    {
        $_totalShopping = ShoppingItem::withParameters(['id_user' => $user_id, 'purchase_date' => $date, 'id_product' => $product_id])->sum('subtotal');

        $_totalPo = POItem::withParameters(['id_user' => $user_id, 'purchase_date' => $date, 'id_product' => $product_id])->sum('subtotal');

        return $_totalShopping - $_totalPo;
    }

    
}
