<?php

namespace App\Services;

use App\Exceptions\ErrorHandlingException;
use App\Http\Requests\SubscriptionCheckoutRequest;
use App\Http\Resources\TransactionSubscriptionResource;
use App\Models\{PaymentMethod, Subscription, TransactionSubscription, UserSubscription};
use App\Services\Xendit\XenditService;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class SubscriptionService
{
    public function __construct(
        private PaymentMethod $payment_method,
        private TransactionSubscription $transaction_subscription,
        private XenditService $xendit_service,
        private NotifService $notif_service,
    ) {
    }
    public function find($id)
    {
        return Subscription::find($id);
    }

    public function find_transaction($id)
    {
        return TransactionSubscription::find($id);
    }

    public function list()
    {
        return Subscription::get()->map(function($item){
                
            $item['expired_at'] = Carbon::now()->addMonth($item->duration)->format('d/m/Y');
            return $item;
        });
    }

    public function checkout(SubscriptionCheckoutRequest $request)
    {
        DB::beginTransaction();

        $invoice_id = getInvoiceId('WEHUB');
        $user_id = auth()->user()->id;

        try {
            $payment_request = $request->payment_request;
            $amount = $request->total_subscription;

            $data_payment_method_fee = $this->payment_method->with('paymentType')->find($payment_request['id']);

            // $fee = calculateFee($data_payment_method_fee->code, $data_payment_method_fee->tax_type, $data_payment_method_fee->tax, $amount);
            $fee = calculateFeeWithoutTax($data_payment_method_fee->code, $data_payment_method_fee->tax_type, $data_payment_method_fee->tax, $amount);
            
            $total_with_fee = $amount + $fee;

            $payment_method_type = $data_payment_method_fee->paymentType?->code;

            $result_json = $this->xendit_service->sendXendit($payment_method_type, $data_payment_method_fee->code, $user_id, $total_with_fee);

            $_storeSubs = $this->transaction_subscription->create([
                'invoice_id'        => $invoice_id, 
                'id_user'           => $user_id, 
                'id_subscription'   => $request->id_subscription, 
                'total'             => $amount,
                'fee'               => $fee, 
                'subtotal'          => $total_with_fee, 
                'status'            => 'waiting',
                'payment_request'   => $data_payment_method_fee,
                'payment_response'  => array_merge($result_json, ['amount' => $amount, 'fee' => $fee, 'total' => $total_with_fee]),
                'payment_limit_at'  => Carbon::now()->addDays(1)
            ]);

            // dd($data);
            $this->notif_service->store("subscribe", "Kode Transkasi: " . $_storeSubs->invoice_id, "Status: ".$_storeSubs->status, 'all', $_storeSubs, 'NotifSubscribeDetail/'.$_storeSubs->id);

            
            DB::commit();

            return $this->transaction_subscription->find($_storeSubs->id);

        } catch (\Throwable $th) {
            DB::rollBack();
            throw new ErrorHandlingException(500, $th->getMessage());
        } catch (\Xendit\Exceptions\ApiException $th) {
            dd($th);
            DB::rollBack();
            throw new ErrorHandlingException(500, $th->getMessage());
        }
    
    }

    public function store($request)
    {
        $id_subscription = $request['id_subscription'];
        $id_user         = $request['id_user'];
        $total           = $request['total'];

        $subs = $this->find($id_subscription);

        $expiredAt = Carbon::now()->addMonth($subs->duration)->toDateString();

        $store = UserSubscription::create([
            'user_id'         => $id_user,
            'subscription_id' => $id_subscription,
            'total_price'     => $total,
            'expired_at'      => $expiredAt
        ]);

        return $store;

    }

    public function check_subscription()
    {
        $now = Carbon::now()->format('Y-m-d H:i:s');

        $trans = TransactionSubscription::withParameters(['status' => 'waiting'])->get();

        foreach ($trans as $value) {
            if ($value->payment_limit_at < $now) {
                $value->update([
                    'status' => 'failed',
                ]);
            }
        }

        return true;
    }

    

    
}
