<?php

namespace App\Services;

use App\Exceptions\{EmailAlreadyTakenException, LoginErrorException, EmailNotFoundException, ErrorHandlingException, UserNotVerifiedException};
use App\Http\Requests\{EmailLoginRequest, ResetRequest, OtpRequest, OtpResendRequest, RegisterRequest};
use App\Models\User;
use App\Models\UserApproved;
use App\Models\UserBussines;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;

class AuthService
{
    public function __construct(
        private OtpService $otp_service
    ) {
        $this->otp_service = $otp_service;
    }

    public function register(RegisterRequest $request)
    {
        DB::beginTransaction();

        $email          = $request->input('email');
        $front_name     = $request->input('front_name');
        $back_name      = $request->input('back_name');
        $phone_number   = $request->input('phone_number');
        $password       = $request->input('password');
        $name_bussines  = $request->input('name_bussines');
        $role           = $request->input('role');

        $_code = $this->get_id($role);
        
        //check verified
        $this->checkVerified($email);

        //check email
        $this->checkEmail($email, $phone_number);

        $user = $this->checkRegister($email) ?? new User();
        $user->code         = $_code;
        $user->role         = $role;
        $user->front_name   = $front_name;
        $user->back_name    = $back_name;
        $user->phone_number = $phone_number;
        $user->email        = $email;
        $user->password     = Hash::make($password);
        $user->skema        = 'bagi_hasil';
        $user->save();

        UserBussines::create([
            'id' => $user->id,
            'name_bussines' => $name_bussines
        ]);

        UserApproved::create([
            'id' => $user->id,
        ]);
        
        #using otp ?

        $otp = $this->otp_service->addOtpRegist($user->id);

        $this->otp_service->sendEmailOtpRegist($email, $otp);

        DB::commit();

        return [
            'data' => [
                'email' => $email
            ]
        ];
    }

    public function forgotPassword($email)
    {
        $user = $this->checkEmailRegister($email);
        if (!$user) {
            throw new EmailNotFoundException();
        }

        $this->otp_service->sendEmailOtpReset($email, $user);

        return [
            'data' => [
                'email' => $email
            ]
        ];
    }

    public function otpResetResend(OtpResendRequest $request)
    {
        $email = $request->input('email');

        $user = $this->checkEmailRegister($email);
        if (!$user) {
            throw new EmailNotFoundException();
        }

        $this->otp_service->deleteOtpReset($user->id);

        $otp = $this->otp_service->addOtpReset($user->id);

        $this->otp_service->sendEmailOtpReset($email, $otp);

        return true;
    }

    public function verifyReset(OtpRequest $request)
    {
        $email = $request->input('email');
        $otp   = $request->input('otp');

        $user = $this->checkEmailRegister($email);
        if (!$user) {
            throw new EmailNotFoundException();
        }

        $this->otp_service->checkOtpVerifyReset($user->id, $otp);

        return [
            'data' => [
                'email' => $email,
                'otp' => $otp
            ]
        ];
    }

    public function resetPassword(ResetRequest $request)
    {
        $email = $request->input('email');

        $user = $this->checkEmailRegister($email);
        if (!$user) {
            throw new EmailNotFoundException();
        }
        
        $password = $request->input('password');

        $update = $user;
        $update->password = Hash::make($password);
        $update->save();

        return true;
    }

    public function checkEmailRegister($email)
    {
        $user = User::where('email', $email)->where('verified_at', '!=', null)->where('active', 'yes')->first();
        if (!$user) {
            throw new EmailNotFoundException();
        }
        return $user;
    }

    public function verifyRegits(OtpRequest $request)
    {
        $email = $request->input('email');
        $otp   = $request->input('otp');

        $user = $this->checkRegister($email);
        if (!$user) {
            throw new EmailNotFoundException();
        }

        $check_otp = $this->otp_service->checkOtpVerifyRegist($user->id, $otp);

        $verified = $user;
        $verified->active = "yes";
        $verified->verified_at = Carbon::now();
        $verified->save();

        $check_otp->delete();

        return $this->redirectLogin($verified);
    }

    public function otpRegistResend(OtpResendRequest $request)
    {
        $email = $request->input('email');

        $user = $this->checkRegister($email);
        if (!$user) {
            throw new EmailNotFoundException();
        }

        $this->otp_service->deleteOtpRegist($user->id);

        $otp = $this->otp_service->addOtpRegist($user->id);

        $this->otp_service->sendEmailOtpRegist($email, $otp);

        return true;
    }

    private function checkRegister($email)
    {
        return User::where('email', $email)->where('verified_at', null)->first();
    }

    private function redirectLogin($user)
    {
        $token = auth()->setTTL(451814)->login($user);

        if ($token) {

            $response_token = json_decode(respondWithToken($token)->content(), true);

            $result = collect(auth()->user());
        } else {

            throw new LoginErrorException();
        }

        return $this->formatLogin($result, $response_token);
    }

    public function checkEmail($email, $phone = "")
    {
        $check = User::where('email', $email)->where('verified_at', '!=', null)->first();
        if ($check) {
            throw new EmailAlreadyTakenException();
        }
    }

    public function emailLogin(EmailLoginRequest $request)
    {
        $credentials = ['email' => $request->input('email'), 'password' => $request->input('password')];

        $token = auth()->setTTL(451814)->attempt($credentials);

        if ($token) {

            $response_token = json_decode(respondWithToken($token)->content(), true);

            $result = collect(auth()->user());

             //check verified
            $this->checkVerified($result['email']);

        } else {

            throw new LoginErrorException();
        }

        return $this->formatLogin($result, $response_token);
    }

    public function checkVerified($email)
    {
        $check = User::where('email', $email)->first();

        if ($check) {

            if ($check->verified_at == "") {

                $otp = $this->otp_service->addOtpRegist($check->id);

                $this->otp_service->sendEmailOtpRegist($check?->email, $otp);

                throw new UserNotVerifiedException($email, $otp);
            }
        }
    }

    private function formatLogin($result, $response_token)
    {
        $data['id'] = $result['id'];
        $data['role'] = $result['role'];
        $data['front_name'] = $result['front_name'];
        $data['back_name'] = $result['back_name'];
        $data['email'] = $result['email'];

        $response['token'] = $response_token['token'];
        $response['token_type'] = $response_token['token_type'];
        $response['success'] = true;
        $response['status'] = 'login';
        $response['message'] = 'login success';
        $response['data'] = $data;

        return $response;
    }

    private function get_id($type)
    {
        $_user = User::where('role', $type)->orderby('code', 'desc')->withTrashed()->first();

        $_id = (int) substr($_user?->code, 6) ?? 0;

        $_type = $type == 'Mitra Kerjasama' ? 'MK' : 'MM';

        $_num = ($_id + 1);

        $_length = Str::length($_num);
        
        $_zero = substr('00000', $_length);
        
        // dd( Carbon::now()->format('my') . $_type . $_zero . $_num);

        return Carbon::now()->format('my') . $_type . $_zero . $_num;
    }
}
