<?php

namespace App\Services;

use App\Http\Requests\CustomerRequest;
use App\Http\Requests\CustomerUpdateRequest;
use App\Http\Requests\ProductRequest;
use App\Http\Requests\ProductUpdateRequest;
use App\Models\{Customer, CustomerAddress};
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use App\Exceptions\ErrorHandlingException;

class CustomerService
{
    public function __construct(
        private UploadService $upload_service,
    ) {
    }

    public function find($id)
    {
        return Customer::find($id);
    }

    public function list($request)
    {
        $user = auth()->user();

        $data = Customer::withParameters(array_merge($request->toArray(), [
            'id_user' => $user->id
        ]))
        ->orderby('id', 'desc');
        $data = $data->latest()->paginate(10);

        $data->getCollection()->transform(function ($item) {

            $show['id']   = $item->id;
            $show['name'] = $item->name;
            $show['pic']  = $item->pic;

            return $show;
        });

        return $data;
    }

    public function create(CustomerRequest $request)
    {
        DB::beginTransaction();

        $user = auth()->user();

        $id = $this->get_id($user);

        $avatar = $request->hasFile('photo') ? $this->upload_service->uploadImage($request->photo, 'customer') : null;
        $contract = $request->hasFile('contract') ? $this->upload_service->uploadImage($request->contract, 'contract') : null;

        $_cust = Customer::create(array_merge($request->all(), [
            'id'            => $id,
            'name'          => $request->name,
            'pic'           => $request->pic,
            'phone_number'  => $request->phone_number,
            'position'      => $request->position,
            'id_user'       => $user->id,
            'avatar'        => $avatar,
            'contract'      => $contract,
        ]));

        CustomerAddress::create([
            'id'            => $_cust->id,
            'id_province'   => $request->id_province,
            'id_city'       => $request->id_city,
            'id_subdistrict' => $request->id_subdistrict,
            'village'       => $request->village,
            'postcode'      => $request->postcode,
            'address'       => $request->address,
            'rt'            => $request->rt,
            'rw'            => $request->rw,
            'latitude'      => $request->latitude,
            'longitude'     => $request->longitude,
        ]);

        DB::commit();

        return true;
    }

    public function update(CustomerUpdateRequest $request)
    {
        DB::beginTransaction();

        $find = $this->find($request->id);

        if ($find) {
            
        // $this->check_number($find->id, $request->phone_number);

        $avatar = $request->hasFile('photo') ? $this->upload_service->uploadImage($request->photo, 'customer') : $find->avatar;
        $contract = $request->hasFile('contract') ? $this->upload_service->uploadImage($request->contract, 'contract') : $find->contract;


        $find->update(array_merge($request->all(), [
            'name'          => $request->name,
            'pic'           => $request->pic,
            'phone_number'  => $request->phone_number,
            'position'      => $request->position,
            'avatar'        => $avatar,
            'contract'      => $contract,
        ]));

        CustomerAddress::updateOrCreate([
            'id'            => $find->id,
        ],[
            'id_province'   => $request->id_province,
            'id_city'       => $request->id_city,
            'id_subdistrict' => $request->id_subdistrict,
            'village'       => $request->village,
            'postcode'      => $request->postcode,
            'address'       => $request->address,
            'rt'            => $request->rt,
            'rw'            => $request->rw,
            'latitude'      => $request->latitude,
            'longitude'     => $request->longitude,
        ]);
        }

        DB::commit();

        return true;
        
    }

    public function delete($id)
    {
        $find = $this->find($id);

        if ($find) {
            $find->delete();
        }

        return true;
    }

    private function check_number($id, $phone)
    {
        $find = Customer::whereNotIn('id', [$id])->where('phone_number', $phone)->first();

        if ($find) {
            throw new ErrorHandlingException(422, 'phone number already use');
        }
    }

    private function get_id($user)
    {
        $_cust = Customer::where('id_user', $user->id)->orderby('created_at', 'desc')->withTrashed()->first();

        $_id = (int) substr($_cust?->id, 18) ?? 0;
        
        return $user->code . 'PEL' . Carbon::now()->format('my') . ($_id + 1);
    }

    
}
