<?php

namespace App\Services;

use App\Http\Requests\PORequest;
use App\Http\Requests\POUpdateRequest;
use App\Http\Requests\ProductRequest;
use App\Http\Requests\ProductUpdateRequest;
use App\Http\Requests\ShoppingRequest;
use App\Http\Requests\ShoppingUpdateRequest;
use App\Models\{PO, POItem};
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class POService
{
    public function __construct(
        private UploadService $upload_service,
        private ProductService $product_service,
        private NotifService $notif_service,
    ) {
    }

    public function find($id)
    {
        return PO::find($id);
    }

    public function findDetail($id)
    {
        return POItem::find($id);
    }

    public function list($request)
    {
        $user = auth()->user();

        $data = PO::with('customer')
        ->withParameters(array_merge($request->toArray(), [
            'id_user' => $user->id
        ]))
        ->orderby('id', 'desc');
        $data = $data->latest()->paginate(10);

        $data->getCollection()->transform(function ($item) {

            $show['id']                  = $item->id;
            $show['purchase_date']       = translatedFormat($item->purchase_date, 'd/m/Y');
            $show['customer']            = $item->customer?->name;
            $show['total_product']       = count($item->items);

            return $show;
        });

        return $data;
    }

    public function get_ar($request)
    {
        $user = auth()->user();

        $data = PO::with('customer')
        ->withParameters(array_merge($request, [
            'id_user' => $user->id,
            'transaction_type' => 'tempo',
            'is_payment' => 'true'
        ]))
        ->orderby('id', 'desc');
        $data = $data->latest()->paginate(10);

        $data->getCollection()->transform(function ($item) {

            $show['id']                  = $item->id;
            $show['due_date']            = translatedFormat($item->due_date, 'd/m/Y');
            $show['customer']            = $item->customer?->name;
            $show['total_payment']       = $item->items?->sum('subtotal');

            return $show;
        });

        return $data;
    }

    public function get_sale()
    {
        $user = auth()->user();

        $data = POItem::with(['po.customer'])
        ->withParameters([
            'id_user' => $user->id
        ])
        ->orderby('id', 'desc');
        $data = $data->latest()->paginate(20);

        $data->getCollection()->transform(function ($item) {

            $_status = $item->po?->payment_date <> '' ? 'Lunas' : 'Belum Lunas';

            $_countDown = countDown($item->po?->due_date, $_status, $item->po?->transaction_type);

            $show['purchase_date']       = translatedFormat($item->po?->purchase_date, 'd/m/Y');
            $show['customer']            = $item->po?->customer?->name;
            $show['product']             = $item->product;
            $show['qty']                 = $item->qty;
            $show['price']               = $item->subtotal / $item->qty;
            $show['subtotal']            = $item->subtotal;
            $show['sale']                = $item->subtotal;
            if ($item->transaction_type == 'tempo') {
                $show['payment_date']        = $item->po?->due_date <> '' ? translatedFormat($item->po?->due_date, 'd/m/Y') : null;
            } else {

                $show['payment_date']        = $item->po?->payment_date <> '' ? translatedFormat($item->po?->payment_date, 'd/m/Y') : null;
            }
            $show['status']              = $_status;
            $show['count_down']          = $_countDown;

            return $show;
        });

        return $data;
    }

    public function payment($request)
    {
        $id = $request->id;

        $upload_payment  = $this->upload_service->uploadImage($request->upload_payment, 'upload_payment');

        $find = $this->find($id);

        if ($find) {
            $find->update([
                'payment_date' => Carbon::now(),
                'upload_payment' => $upload_payment
            ]);
        }

        $this->notif_service->store("po", "Kode Transkasi: " . $find->id, "telah dibayarkan", "all", $find);

        return true;
    }

    public function create(PORequest $request)
    {
        DB::beginTransaction();

        $user = auth()->user();

        $_id = $this->get_id($user);

        $upload_po       = $this->upload_service->uploadImage($request->upload_po, 'upload_po');
        $upload_handover = $this->upload_service->uploadImage($request->upload_handover, 'upload_handover');
        $upload_receipt  = $this->upload_service->uploadImage($request->upload_receipt, 'upload_receipt');

        $po = PO::create(array_merge($request->except('products'), [
            'id'            => $_id,
            'id_user'       => $user->id,
            'upload_handover'=> $upload_handover,
            'upload_po'     => $upload_po,
            'upload_receipt'=> $upload_receipt,
            'payment_date'  => $request->transaction_type == 'tunai' ? $request->purchase_date : null,
        ]));

        $products = $request->products;

        foreach ($products as $product) {
            $find = $this->product_service->find($product['id_product']);

            $_arrayproduct = [
                'title' => $find?->title,
                'sku' => $find?->sku,
                'category' => $find?->category?->title,
                'unit' => $find?->unit?->title,
                'package' => $find?->package?->title,
            ];

            POItem::create([ 
                'id_po'         => $po->id,
                'id_product'    => $product['id_product'],
                'product'       => $_arrayproduct,
                'qty'           => $product['qty'],
                'subtotal'      => $product['subtotal'],
                'description'   => $product['description'],
            ]);
        }

        DB::commit();

        return true;
    }

    public function update(POUpdateRequest $request)
    {
        DB::beginTransaction();
        
        $find = $this->find($request->id);

        if ($find) {
            $upload_po       = $request->hasFile('upload_po') ? $this->upload_service->uploadImage($request->upload_po, 'upload_po'): $find->upload_po;
            $upload_handover = $request->hasFile('upload_handover') ? $this->upload_service->uploadImage($request->upload_handover, 'upload_handover'): $find->upload_handover;
            $upload_receipt  = $request->hasFile('upload_receipt') ? $this->upload_service->uploadImage($request->upload_receipt, 'upload_receipt'): $find->upload_receipt;

            $find->update(array_merge($request->except('products'), [
                'upload_handover'=> $upload_handover,
                'upload_po'     => $upload_po,
                'upload_receipt'=> $upload_receipt,
                'payment_date'  => $request->transaction_type == 'tunai' ? $request->purchase_date : $find->payment_date,
            ]));

            $products = $request->products;

            $_id = [];

            foreach ($products as $product) {
                if ($product['id'] != "") {
                    $_item = $this->findDetail($product['id']);

                    $_id[] = $product['id'];

                    $_product = $this->product_service->find($product['id_product']);

    
                    $_arrayproduct = [
                        'title' => $_product?->title,
                        'sku' => $_product?->sku,
                        'category' => $_product?->category?->title,
                        'unit' => $_product?->unit?->title,
                        'package' => $_product?->package?->title,
                    ];
    
                    $_item->update([
                        'product'       => $_arrayproduct,
                        'qty'           => $product['qty'],
                        'subtotal'      => $product['subtotal'],
                        'description'   => $product['description'],
                    ]);

                }
            }

            POItem::whereNotIn('id', $_id)->where('id_po', $find->id)->delete();

            foreach ($products as $product) {
                if ($product['id'] == "") {
                    $_product = $this->product_service->find($product['id_product']);

                    $_arrayproduct = [
                        'title' => $_product?->title,
                        'sku' => $_product?->sku,
                        'category' => $_product?->category?->title,
                        'unit' => $_product?->unit?->title,
                        'package' => $_product?->package?->title,
                    ];
        
                    POItem::create([
                        'id_po'         => $find->id,
                        'id_product'    => $product['id_product'],
                        'product'       => $_arrayproduct,
                        'qty'           => $product['qty'],
                        'subtotal'      => $product['subtotal'],
                        'description'   => $product['description'],
                    ]);

                }
            }
        
        }
        DB::commit();

        return true;
        
    }

    private function get_id($user)
    {
        $_po = PO::where('id_user', $user->id)->latest()->withTrashed()->first();

        $_id = (int) substr($_po?->id, 17) ?? 0;

        return $user->code . 'PO' . Carbon::now()->format('my') . ($_id + 1);
    }

    public function delete($id)
    {
        $find = $this->find($id);

        if ($find) {
            $find->delete();
        }

        return true;
    }

    
}
