<?php

namespace App\Services\Venture;

use App\Models\Venture\User;
use Illuminate\Http\Request;
use InvalidArgumentException;
use App\Models\Venture\Account;
use App\Models\Venture\Approve;
use App\Models\Venture\Bussines;
use App\Models\Venture\Saldo;
use App\Services\PartialService;
use Illuminate\Support\Facades\DB;
use App\Services\ValidationService;
use Illuminate\Support\Facades\Hash;
use App\Repositories\Venture\VentureRepository;

class VentureService {

  private $fillableUser;
  private $fillableAccount;
  private $fillableBussines;

  public function __construct(
    private PartialService $partial,
    private VentureRepository $repository,
    private ValidationService $validation
  ) {
    $this->fillableUser = [
      'front_name', 'back_name', 'nik', 'phone_number', 'email', 'id_profession'
    ];

    $this->fillableAccount = [
      'bank', 'branch', 'account_number'
    ];

    $this->fillableBussines = [
      'name_bussines', 'id_province', 'id_city', 'id_subdistrict', 'village', 'postcode', 'address', 'latitude', 'longitude'
    ];
  }

  function store(Request $req){
    $this->validation->venture($req, 'store');
 
    $user = User::create(array_merge($req->only($this->fillableUser), [
      'password' => Hash::make($req->password),
      'role' => $req->role
    ]));

    $user = $this->repository->user($user->id);
    $user->update([
      'profile' => $this->partial->upload($req->profile, "users/$user->id/profile"),
      'ktp' => $this->partial->upload($req->ktp, "users/$user->id/ktp"),
      'npwp' => $this->partial->upload($req->npwp, "users/$user->id/npwp")
    ]);

    Account::create(array_merge($req->only($this->fillableAccount), [
      'id' => $user->id,
      'names' => $req->account_name      
    ]));

    Bussines::create(array_merge($req->only($this->fillableBussines), [
      'id' => $user->id,
      'profile_bussines' => $this->partial->upload($req->profile_bussines, "users/$user->id/profile_bussines"),
      'akta' => $this->partial->upload($req->akta, "users/$user->id/akta"),
      'npwp' => $this->partial->upload($req->npwp_bussines, "users/$user->id/npwp_bussines"),
      'nib' => $this->partial->upload($req->nib, "users/$user->id/nib"),
      'sk_kemenkum' => $this->partial->upload($req->sk_kemenkum, "users/$user->id/sk_kemenkum")
    ])); 

    Approve::create([
      'id' => $user->id,
    ]);

    return [
      'message' => config('message.ADD_SUCCESS'),
      'url' => url('ventures')
    ];
  }

  function update(Request $req){
    $this->validation->venture($req, 'update');

    $user = $this->repository->user($req->id);
    if (!$user) {
      throw new InvalidArgumentException(config('message.DATA_NOT_FOUND'), 404);
    }
    $_profile = $user->profile;
    $_ktp = $user->ktp;
    $_npwp = $user->npwp;

    $account = $this->repository->account($req->id);
    if (!$account) {
      throw new InvalidArgumentException(config('message.DATA_NOT_FOUND'), 404);
    }

    $bussines = $this->repository->bussines($req->id);
    if (!$bussines) {
      throw new InvalidArgumentException(config('message.DATA_NOT_FOUND'), 404);
    }
    $_profile_bussines = $bussines->profile_bussines;
    $_akta = $bussines->akta;
    $_npwp_bussines = $bussines->npwp_bussines;
    $_nib = $bussines->nib;
    $_sk_kemenkum = $bussines->sk_kemenkum;

    $user->update(array_merge($req->only($this->fillableUser), [
      'password' => $req->password ? Hash::make($req->password) : $user->password,
      'profile' => $this->partial->upload($req->profile, "users/$req->id/profile", $_profile),
      'ktp' => $this->partial->upload($req->ktp, "users/$req->id/ktp", $_ktp),
      'npwp' => $this->partial->upload($req->npwp, "users/$req->id/npwp", $_npwp)
    ]));

    $account->update(array_merge($req->only($this->fillableAccount), [
      'names' => $req->account_name      
    ]));

    $bussines->update(array_merge($req->only($this->fillableBussines), [
      'profile_bussines' => $this->partial->upload($req->profile_bussines, "users/$req->id/profile_bussines", $_profile_bussines),
      'akta' => $this->partial->upload($req->akta, "users/$req->id/akta", $_akta),
      'npwp' => $this->partial->upload($req->npwp_bussines, "users/$req->id/npwp_bussines", $_npwp_bussines),
      'nib' => $this->partial->upload($req->nib, "users/$req->id/nib", $_nib),
      'sk_kemenkum' => $this->partial->upload($req->sk_kemenkum, "users/$req->id/sk_kemenkum", $_sk_kemenkum)
    ]));

    return [
      'message' => config('message.UPDATE_SUCCESS'),
      'url' => url('ventures')
    ];
  }

  function delete(Request $req){
    $find = $this->repository->user($req->id);

    if (!$find) {
      throw new InvalidArgumentException(config('message.DATA_NOT_FOUND'), 404);
    }

    $find->delete();

    return [
      'message' => config('message.DELETE_SUCCESS'),
    ];
  }

  function verification(Request $req){
    $find = $this->repository->user($req->id);

    if (!$find) {
      throw new InvalidArgumentException(config('message.DATA_NOT_FOUND'), 404);
    }

    $approve = $this->repository->approve($req->id);

    $approve->update([
      'id_admin' => auth()->user()->id,
      'until' => $req->until,
      'status' => 'ya'
    ]);

    return [
      'message' => 'Berhasil melakukan verifikasi!',
      'url' => url('ventures')
    ];
  }

  function cooperation(Request $req){
    $find = $this->repository->user($req->id);

    if (!$find || $find->role != 'Mitra Kerjasama') {
      throw new InvalidArgumentException(config('message.DATA_NOT_FOUND'), 404);
    }

    $share_portion = $req->share_portion ? str_replace(',', '.', $req->share_portion) : $find->share_portion;

    $find->update([
      'share_portion' => $share_portion,
      'skema' => $req->skema ?? $find->skema,
      'pks' => $req->pks ? $this->partial->upload($req->pks, 'pks') : $find->pks
    ]);

    return [
      'message' => 'Berhasil mempebarui status kerjasama!',
    ];
  }

  function wallet(Request $req){
    $this->validation->wallet($req, 'store');

    $find = $this->repository->user($req->user_id);

    if (!$find || $find->role != 'Mitra Kerjasama') {
      throw new InvalidArgumentException(config('message.DATA_NOT_FOUND'), 404);
    }

    $setor = preg_replace('/[^0-9]/', '', $req->setor);

    Saldo::create(array_merge($req->only(['user_id', 'modal', 'periode']), [
      'setor' => $setor,
      'file' => $this->partial->upload($req->file, 'modal_setor'),
      'result' => $setor * ($req->modal / 100)
    ]));

    return [
      'message' => 'Berhasil menambah history!',
      'url' => 'reload'
    ];
  }

  function saldo(Request $req){
    $find = $this->repository->user($req->user_id);

    if (!$find || $find->role != 'Mitra Kerjasama') {
      throw new InvalidArgumentException(config('message.DATA_NOT_FOUND'), 404);
    }

    $find->update([
      'saldo' => preg_replace('/[^0-9]/', '', $req->saldo)
    ]);

    return [
      'message' => 'Berhasil mengubah saldo!',
      'url' => 'reload'
    ];
  }

  function container(Request $req, $action){
    DB::beginTransaction();

    try {

      if ($action == 'store') {
        $e = $this->store($req);
      } elseif ($action == 'update') {
        $e = $this->update($req);
      } elseif ($action == 'delete') {
        $e = $this->delete($req);
      } elseif ($action == 'verification') {
        $e = $this->verification($req);
      } elseif ($action == 'cooperation') {
        $e = $this->cooperation($req);
      } elseif ($action == 'wallet') {
        $e = $this->wallet($req);
      } elseif ($action == 'saldo') {
        $e = $this->saldo($req);
      }

      DB::commit();

      return response()->json(array_merge($e, [
        'code' => 200,
        'success' => true,
      ]));
    } catch (\Throwable $th) {

      DB::rollBack();
      
      return response()->json([
        'code' => 500,
        'success' => false,
        'message' => $th->getMessage()
      ], 500);
    }
  }

}